
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; MODULE      : mupad-input.scm
;; DESCRIPTION : MuPAD input converters
;; COPYRIGHT   : (C) 2003 MuPAD research group & SciFace Software GmbH
;;
;; This software falls under the GNU general public license version 3 or later.
;; It comes WITHOUT ANY WARRANTY WHATSOEVER. For details, see the file LICENSE
;; in the root directory or <http://www.gnu.org/licenses/gpl-3.0.html>.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(texmacs-module (mupad-input)
  (:use (utils plugins plugin-convert)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Specific conversion routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (mupad-input-var-row r)
  (if (nnull? r)
      (begin
	(display ", ")
	(plugin-input (car r))
	(mupad-input-var-row (cdr r)))))

(define (mupad-input-row r)
  (display "[")
  (plugin-input (car r))
  (mupad-input-var-row (cdr r))
  (display "]"))

(define (mupad-input-var-rows t)
  (if (nnull? t)
      (begin
	(display ", ")
	(mupad-input-row (car t))
	(mupad-input-var-rows (cdr t)))))

(define (mupad-input-rows t)
  (display "matrix([")
  (mupad-input-row (car t))
  (mupad-input-var-rows (cdr t))
  (display "])"))

(define (mupad-input-descend-last args)
  (if (null? (cdr args))
      (plugin-input (car args))
      (mupad-input-descend-last (cdr args))))

(define (mupad-input-det args)
  (display "linalg::det(")
  (mupad-input-descend-last args)
  (display ")"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; big operators
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (mupad-input-big args)
  (if (== (car args) ".")
      (display ")")
      (begin
	(display "(")
	(if (== (car args) "prod")
	    (display "product")
	    (display (car args)))
        (display "@stdlib::swapargs)(")
        (if (nnull? (cdr args))
            (begin
              (plugin-input (cadr args))
	      (if (null? (cddr args))
		  (display ",")
		  (begin (display "..")
			 (plugin-input (caddr args))
			 (display ","))))))))

(define (mupad-input-big-around args)
  (let* ((b `(big-around ,@args))
	 (op (big-name b))
	 (sub (big-subscript b))
	 (sup (big-supscript b))
	 (body (big-body b))
	 (l (cond ((and sub sup) (list op sub sup))
		  (sub (list op sub))
		  (else (list op)))))
    (mupad-input-big l)
    (plugin-input body)
    (display ")")))

;; TODO: Check for the type of primes.
(define (mupad-input-rprime args)
  (display (car args)))

;; TODO: Check for the type of operator
(define (mupad-input-mathbin args)
  (display " ")
  (plugin-input (car args))
  (display " "))

(define (mupad-input-sqrt args)
  (if (= (length args) 1)
      (begin
	(display "sqrt(")
	(plugin-input (car args))
	(display ")"))
      (begin
	(display "surd(")
	(plugin-input (car args))
	(display ",")
	(plugin-input (cadr args))
	(display ")"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; decorations
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define mupad-inside-identifier nil)

(define (mupad-input-wide args)
  (if (== mupad-inside-identifier nil)
      (display "`"))
  (let ((bot (car args)) (acc (cadr args)) (inident mupad-inside-identifier))
    (set! mupad-inside-identifier t)
    (plugin-input bot)
    (cond ((== acc "^")           (display "&circ;"))
	  ((== acc "~")           (display "&tilde;"))
; <bind>
; <abovering>
	  ((== acc "<bar>")       (display "&OverBar;"))
	  ((== acc "<wide-bar>")  (display "&OverBar;"))
	  ((== acc "<vect>")      (display "&rarr;"))
          ((== acc "<check>")     (display "&check;"))
          ((== acc "<acute>")     (display "&acute;"))
          ((== acc "<grave>")     (display "&grave;"))
          ((== acc "<breve>")     (display "&breve;"))
          ((== acc "<dot>")       (display "&dot;"))
          ((== acc "<ddot>")      (display "&die;"))
          ((== acc "<dddot>")     (display "&tdot;"))
;          ((== acc "<ddddot>")    (display "&ddie;"))
	  (else 
	   (display " acc ")
	   (plugin-input acc)))
    (set! mupad-inside-identifier inident))
  (if (== mupad-inside-identifier nil)
      (display "`"))
  )

(define (mupad-input-wide-under args)
  (if (== mupad-inside-identifier nil)
      (display "`"))
  (let ((bot (car args)) (acc (cadr args)) (inident mupad-inside-identifier))
    (set! mupad-inside-identifier t)
    (plugin-input bot)
;    (cond ((== acc "^")           (display "&circ;"))
;	  ((== acc "~")           (display "&tilde;"))
;	  ((== acc "<bar>")       (display "&UnderBar;"))
;	  ((== acc "<wide-bar>")  (display "&UnderBar;"))
;	  ((== acc "<vect>")      (display "&rarr;"))
;          ((== acc "<check>")     (display "&check;"))
;          ((== acc "<acute>")     (display "&acute;"))
;          ((== acc "<grave>")     (display "&grave;"))
;          ((== acc "<breve>")     (display "&breve;"))
;          ((== acc "<dot>")       (display "&dot;"))
;          ((== acc "<ddot>")      (display "&die;"))
;          ((== acc "<dddot>")     (display "&tdot;"))
;;          ((== acc "<ddddot>")    (display "&ddie;"))
;	  (else 
	   (display " acc below ")
	   (plugin-input acc)
;))
    (set! mupad-inside-identifier inident))
  (if (== mupad-inside-identifier nil)
      (display "`"))
  )

(define (mupad-send-identifier arg)
;(display "/* ") (display arg) (display " */")
  (if (== mupad-inside-identifier nil)
      (display "`"))
  (let ((inident mupad-inside-identifier))
    (set! mupad-inside-identifier t)
    (cond
     ((== arg "<alpha>")      (display "&alpha;"))
     ((== arg "<beta>")       (display "&beta;"))
     ((== arg "<gamma>")      (display "&gamma;"))
     ((== arg "<delta>")      (display "&delta;"))
     ((== arg "<epsilon>")    (display "&epsilon;"))
     ((== arg "<varepsilon>") (display "&epsiv;"))
     ((== arg "<zeta>")       (display "&zeta;"))
     ((== arg "<eta>")        (display "&eta;"))
     ((== arg "<theta>")      (display "&theta;"))
     ((== arg "<vartheta>")   (display "&thetav;"))
     ((== arg "<iota>")       (display "&iota;"))
     ((== arg "<kappa>")      (display "&kappa;"))
     ((== arg "<lambda>")     (display "&lambda;"))
     ((== arg "<mu>")         (display "&mu;"))
     ((== arg "<nu>")         (display "&nu;"))
     ((== arg "<xi>")         (display "&xi;"))
     ((== arg "<omicron>")    (display "&omicron;"))
     ((== arg "<pi>")         (display "&pi;"))
     ((== arg "<varpi>")      (display "&piv;"))
     ((== arg "<rho>")        (display "&rho;"))
     ((== arg "<varrho>")     (display "&rhov;"))
     ((== arg "<sigma>")      (display "&sigma;"))
     ((== arg "<varsigma>")   (display "&sigmav;"))
     ((== arg "<tau>")        (display "&tau;"))
     ((== arg "<upsilon>")    (display "&upsilon;"))
     ((== arg "<phi>")        (display "&phi;"))
     ((== arg "<varphi>")     (display "&phiv;"))
     ((== arg "<chi>")        (display "&chi;"))
     ((== arg "<psi>")        (display "&psi;"))
     ((== arg "<omega>")      (display "&omega;"))
     
     ((== arg "<Gamma>")      (display "&Gamma;"))
     ((== arg "<Delta>")      (display "&Delta;"))
     ((== arg "<Theta>")      (display "&Theta;"))
     ((== arg "<Lambda>")     (display "&Lambda;"))
     ((== arg "<Xi>")         (display "&Xi;"))
     ((== arg "<Pi>")         (display "&Pi;"))
     ((== arg "<Sigma>")      (display "&Sigma;"))
     ((== arg "<Upsilon>")    (display "&Upsilon;"))
     ((== arg "<Phi>")        (display "&Phi;"))
     ((== arg "<Psi>")        (display "&Psi;"))
     ((== arg "<Omega>")      (display "&Omega;"))

     (else 
      (if (and (!= arg "") (== (string-ref arg 0) #\<))
	  (if (and (> (string-length arg) 3)
		   (== (substring arg 0 3) "<b-"))
	      (begin
		(mupad-send-identifier 
		(string-append "<" 
			       (substring arg 3 
					  (string-length arg))))
	       (display "&bold;"))
	      (if (and (> (string-length arg) 5)
		       (== (substring arg 0 5) "<cal-"))
		  (begin
		    (mupad-send-identifier 
		     (string-append "<" 
				    (substring arg 5
					       (string-length arg))))
		    (display "&cal;"))
		  (if (and (> (string-length arg) 6)
			   (== (substring arg 0 6) "<frak-"))
		      (begin
			(mupad-send-identifier 
			 (string-append "<" 
					(substring arg 6
						   (string-length arg))))
			(display "&frak;"))
		      (if (and (> (string-length arg) 5)
			       (== (substring arg 0 5) "<bbb-"))
			  (begin
			    (mupad-send-identifier 
			     (string-append "<" 
					    (substring arg 5
						       (string-length arg))))
			    (display "&Bbb;"))
			  (display (substring arg 1 (- (string-length arg) 1)))))))
	  (display arg)))
     )
    (set! mupad-inside-identifier inident))
  (if (== mupad-inside-identifier nil)
      (display "`"))
  )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Initialization
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(plugin-input-converters mupad
  (rows mupad-input-rows)
  (det mupad-input-det)
  (big-around mupad-input-big-around)
  (wide mupad-input-wide)
  (wide* mupad-input-wide-under)
  (rprime mupad-input-rprime)
  (sqrt mupad-input-sqrt)

  (mathbin mupad-input-mathbin)

  ("<mapsto>"     " -<gtr> ")
  ("<longmapsto>" " -<gtr> ")
  ("<cdot>"       " * ")
  ("<ast>"        " * ")
  ("<div>"        " / ")
  ("<circ>"       "@")
  ("<cap>"        " intersect ")
  ("<cup>"        " union ")
  ("<uplus>"      " union ")
  ("<veebar>"     " xor ")
  ("<in>"         " in ")
  ("<subset>"     " subset ")
  ("<subseteq>"   " subset ")
  ("<leqq>"       " <less>= ")
  ("<geqq>"       " <gtr>= ")
  ("<neq>"        " <less><gtr> ")
  ("<ldots>"      "...")

  ("<Rightarrow>"         "==<gtr>")
  ("<Longrightarrow>"     "==<gtr>")
  ("<Leftrightarrow>"     "<less>=<gtr>")
  ("<Longleftrightarrow>" "<less>=<gtr>")

;; This line makes *all* input go through mupad-send-identifier
;  ('identifier mupad-send-identifier)

;  ("<alpha>"      "`&alpha;`")
  ("<alpha>"      mupad-send-identifier)
  ("<beta>"       mupad-send-identifier)
  ("<gamma>"      mupad-send-identifier)
  ("<delta>"      mupad-send-identifier)
  ("<epsilon>"    mupad-send-identifier)
  ("<varepsilon>" mupad-send-identifier)
  ("<zeta>"       mupad-send-identifier)
  ("<eta>"        mupad-send-identifier)
  ("<theta>"      mupad-send-identifier)
  ("<vartheta>"   mupad-send-identifier)
  ("<iota>"       mupad-send-identifier)
  ("<kappa>"      mupad-send-identifier)
  ("<lambda>"     mupad-send-identifier)
  ("<mu>"         mupad-send-identifier)
  ("<nu>"         mupad-send-identifier)
  ("<xi>"         mupad-send-identifier)
  ("<omicron>"    mupad-send-identifier)
  ("<pi>"         mupad-send-identifier)
  ("<varpi>"      mupad-send-identifier)
  ("<rho>"        mupad-send-identifier)
  ("<varrho>"     mupad-send-identifier)
  ("<sigma>"      mupad-send-identifier)
  ("<varsigma>"   mupad-send-identifier)
  ("<tau>"        mupad-send-identifier)
  ("<upsilon>"    mupad-send-identifier)
  ("<phi>"        mupad-send-identifier)
  ("<varphi>"     mupad-send-identifier)
  ("<chi>"        mupad-send-identifier)
  ("<psi>"        mupad-send-identifier)
  ("<omega>"      mupad-send-identifier)
  ("<Gamma>"      mupad-send-identifier)
  ("<Delta>"      mupad-send-identifier)
  ("<Theta>"      mupad-send-identifier)
  ("<Lambda>"     mupad-send-identifier)
  ("<Xi>"         mupad-send-identifier)
  ("<Pi>"         mupad-send-identifier)
  ("<Sigma>"      mupad-send-identifier)
  ("<Upsilon>"    mupad-send-identifier)
  ("<Phi>"        mupad-send-identifier)
  ("<Psi>"        mupad-send-identifier)
  ("<Omega>"      mupad-send-identifier)

  ("<mathi>"    " I ")
  ("<mathe>"    " E ")
  ("<mathpi>"   " PI ")
  ("<emptyset>" " {} ")
  ("<neg>"      " not ")

  ("<b-0>" mupad-send-identifier)
  ("<b-1>" mupad-send-identifier)
  ("<b-2>" mupad-send-identifier)
  ("<b-3>" mupad-send-identifier)
  ("<b-4>" mupad-send-identifier)
  ("<b-5>" mupad-send-identifier)
  ("<b-6>" mupad-send-identifier)
  ("<b-7>" mupad-send-identifier)
  ("<b-8>" mupad-send-identifier)
  ("<b-9>" mupad-send-identifier)
  
  ("<b-a>" mupad-send-identifier)
  ("<b-b>" mupad-send-identifier)
  ("<b-c>" mupad-send-identifier)
  ("<b-d>" mupad-send-identifier)
  ("<b-e>" mupad-send-identifier)
  ("<b-f>" mupad-send-identifier)
  ("<b-g>" mupad-send-identifier)
  ("<b-h>" mupad-send-identifier)
  ("<b-i>" mupad-send-identifier)
  ("<b-j>" mupad-send-identifier)
  ("<b-k>" mupad-send-identifier)
  ("<b-l>" mupad-send-identifier)
  ("<b-m>" mupad-send-identifier)
  ("<b-n>" mupad-send-identifier)
  ("<b-o>" mupad-send-identifier)
  ("<b-p>" mupad-send-identifier)
  ("<b-q>" mupad-send-identifier)
  ("<b-r>" mupad-send-identifier)
  ("<b-s>" mupad-send-identifier)
  ("<b-t>" mupad-send-identifier)
  ("<b-u>" mupad-send-identifier)
  ("<b-v>" mupad-send-identifier)
  ("<b-w>" mupad-send-identifier)
  ("<b-x>" mupad-send-identifier)
  ("<b-y>" mupad-send-identifier)
  ("<b-z>" mupad-send-identifier)
  ("<b-A>" mupad-send-identifier)
  ("<b-B>" mupad-send-identifier)
  ("<b-C>" mupad-send-identifier)
  ("<b-D>" mupad-send-identifier)
  ("<b-E>" mupad-send-identifier)
  ("<b-F>" mupad-send-identifier)
  ("<b-G>" mupad-send-identifier)
  ("<b-H>" mupad-send-identifier)
  ("<b-I>" mupad-send-identifier)
  ("<b-J>" mupad-send-identifier)
  ("<b-K>" mupad-send-identifier)
  ("<b-L>" mupad-send-identifier)
  ("<b-M>" mupad-send-identifier)
  ("<b-N>" mupad-send-identifier)
  ("<b-O>" mupad-send-identifier)
  ("<b-P>" mupad-send-identifier)
  ("<b-Q>" mupad-send-identifier)
  ("<b-R>" mupad-send-identifier)
  ("<b-S>" mupad-send-identifier)
  ("<b-T>" mupad-send-identifier)
  ("<b-U>" mupad-send-identifier)
  ("<b-V>" mupad-send-identifier)
  ("<b-W>" mupad-send-identifier)
  ("<b-X>" mupad-send-identifier)
  ("<b-Y>" mupad-send-identifier)
  ("<b-Z>" mupad-send-identifier)
  
  ("<b-alpha>"      mupad-send-identifier)
  ("<b-beta>"       mupad-send-identifier)
  ("<b-gamma>"      mupad-send-identifier)
  ("<b-delta>"      mupad-send-identifier)
  ("<b-epsilon>"    mupad-send-identifier)
  ("<b-varepsilon>" mupad-send-identifier)
  ("<b-zeta>"       mupad-send-identifier)
  ("<b-eta>"        mupad-send-identifier)
  ("<b-theta>"      mupad-send-identifier)
  ("<b-vartheta>"   mupad-send-identifier)
  ("<b-iota>"       mupad-send-identifier)
  ("<b-kappa>"      mupad-send-identifier)
  ("<b-lambda>"     mupad-send-identifier)
  ("<b-mu>"         mupad-send-identifier)
  ("<b-nu>"         mupad-send-identifier)
  ("<b-xi>"         mupad-send-identifier)
  ("<b-omicron>"    mupad-send-identifier)
  ("<b-pi>"         mupad-send-identifier)
  ("<b-varpi>"      mupad-send-identifier)
  ("<b-rho>"        mupad-send-identifier)
  ("<b-varrho>"     mupad-send-identifier)
  ("<b-sigma>"      mupad-send-identifier)
  ("<b-varsigma>"   mupad-send-identifier)
  ("<b-tau>"        mupad-send-identifier)
  ("<b-upsilon>"    mupad-send-identifier)
  ("<b-phi>"        mupad-send-identifier)
  ("<b-varphi>"     mupad-send-identifier)
  ("<b-chi>"        mupad-send-identifier)
  ("<b-psi>"        mupad-send-identifier)
  ("<b-omega>"      mupad-send-identifier)
  ("<b-Gamma>"      mupad-send-identifier)
  ("<b-Delta>"      mupad-send-identifier)
  ("<b-Theta>"      mupad-send-identifier)
  ("<b-Lambda>"     mupad-send-identifier)
  ("<b-Xi>"         mupad-send-identifier)
  ("<b-Pi>"         mupad-send-identifier)
  ("<b-Sigma>"      mupad-send-identifier)
  ("<b-Upsilon>"    mupad-send-identifier)
  ("<b-Phi>"        mupad-send-identifier)
  ("<b-Psi>"        mupad-send-identifier)
  ("<b-Omega>"      mupad-send-identifier)

  ("<cal-A>" mupad-send-identifier)
  ("<cal-B>" mupad-send-identifier)
  ("<cal-C>" mupad-send-identifier)
  ("<cal-D>" mupad-send-identifier)
  ("<cal-E>" mupad-send-identifier)
  ("<cal-F>" mupad-send-identifier)
  ("<cal-G>" mupad-send-identifier)
  ("<cal-H>" mupad-send-identifier)
  ("<cal-I>" mupad-send-identifier)
  ("<cal-J>" mupad-send-identifier)
  ("<cal-K>" mupad-send-identifier)
  ("<cal-L>" mupad-send-identifier)
  ("<cal-M>" mupad-send-identifier)
  ("<cal-N>" mupad-send-identifier)
  ("<cal-O>" mupad-send-identifier)
  ("<cal-P>" mupad-send-identifier)
  ("<cal-Q>" mupad-send-identifier)
  ("<cal-R>" mupad-send-identifier)
  ("<cal-S>" mupad-send-identifier)
  ("<cal-T>" mupad-send-identifier)
  ("<cal-U>" mupad-send-identifier)
  ("<cal-V>" mupad-send-identifier)
  ("<cal-W>" mupad-send-identifier)
  ("<cal-X>" mupad-send-identifier)
  ("<cal-Y>" mupad-send-identifier)
  ("<cal-Z>" mupad-send-identifier)

  ("<frak-a>" mupad-send-identifier)
  ("<frak-b>" mupad-send-identifier)
  ("<frak-c>" mupad-send-identifier)
  ("<frak-d>" mupad-send-identifier)
  ("<frak-e>" mupad-send-identifier)
  ("<frak-f>" mupad-send-identifier)
  ("<frak-g>" mupad-send-identifier)
  ("<frak-h>" mupad-send-identifier)
  ("<frak-i>" mupad-send-identifier)
  ("<frak-j>" mupad-send-identifier)
  ("<frak-k>" mupad-send-identifier)
  ("<frak-l>" mupad-send-identifier)
  ("<frak-m>" mupad-send-identifier)
  ("<frak-n>" mupad-send-identifier)
  ("<frak-o>" mupad-send-identifier)
  ("<frak-p>" mupad-send-identifier)
  ("<frak-q>" mupad-send-identifier)
  ("<frak-r>" mupad-send-identifier)
  ("<frak-s>" mupad-send-identifier)
  ("<frak-t>" mupad-send-identifier)
  ("<frak-u>" mupad-send-identifier)
  ("<frak-v>" mupad-send-identifier)
  ("<frak-w>" mupad-send-identifier)
  ("<frak-x>" mupad-send-identifier)
  ("<frak-y>" mupad-send-identifier)
  ("<frak-z>" mupad-send-identifier)
  ("<frak-A>" mupad-send-identifier)
  ("<frak-B>" mupad-send-identifier)
  ("<frak-C>" mupad-send-identifier)
  ("<frak-D>" mupad-send-identifier)
  ("<frak-E>" mupad-send-identifier)
  ("<frak-F>" mupad-send-identifier)
  ("<frak-G>" mupad-send-identifier)
  ("<frak-H>" mupad-send-identifier)
  ("<frak-I>" mupad-send-identifier)
  ("<frak-J>" mupad-send-identifier)
  ("<frak-K>" mupad-send-identifier)
  ("<frak-L>" mupad-send-identifier)
  ("<frak-M>" mupad-send-identifier)
  ("<frak-N>" mupad-send-identifier)
  ("<frak-O>" mupad-send-identifier)
  ("<frak-P>" mupad-send-identifier)
  ("<frak-Q>" mupad-send-identifier)
  ("<frak-R>" mupad-send-identifier)
  ("<frak-S>" mupad-send-identifier)
  ("<frak-T>" mupad-send-identifier)
  ("<frak-U>" mupad-send-identifier)
  ("<frak-V>" mupad-send-identifier)
  ("<frak-W>" mupad-send-identifier)
  ("<frak-X>" mupad-send-identifier)
  ("<frak-Y>" mupad-send-identifier)
  ("<frak-Z>" mupad-send-identifier)

  ("<bbb-a>" mupad-send-identifier)
  ("<bbb-b>" mupad-send-identifier)
  ("<bbb-c>" mupad-send-identifier)
  ("<bbb-d>" mupad-send-identifier)
  ("<bbb-e>" mupad-send-identifier)
  ("<bbb-f>" mupad-send-identifier)
  ("<bbb-g>" mupad-send-identifier)
  ("<bbb-h>" mupad-send-identifier)
  ("<bbb-i>" mupad-send-identifier)
  ("<bbb-j>" mupad-send-identifier)
  ("<bbb-k>" mupad-send-identifier)
  ("<bbb-l>" mupad-send-identifier)
  ("<bbb-m>" mupad-send-identifier)
  ("<bbb-n>" mupad-send-identifier)
  ("<bbb-o>" mupad-send-identifier)
  ("<bbb-p>" mupad-send-identifier)
  ("<bbb-q>" mupad-send-identifier)
  ("<bbb-r>" mupad-send-identifier)
  ("<bbb-s>" mupad-send-identifier)
  ("<bbb-t>" mupad-send-identifier)
  ("<bbb-u>" mupad-send-identifier)
  ("<bbb-v>" mupad-send-identifier)
  ("<bbb-w>" mupad-send-identifier)
  ("<bbb-x>" mupad-send-identifier)
  ("<bbb-y>" mupad-send-identifier)
  ("<bbb-z>" mupad-send-identifier)
  ("<bbb-A>" mupad-send-identifier)
  ("<bbb-B>" mupad-send-identifier)
  ("<bbb-C>" "C_")
  ("<bbb-D>" mupad-send-identifier)
  ("<bbb-E>" mupad-send-identifier)
  ("<bbb-F>" mupad-send-identifier)
  ("<bbb-G>" mupad-send-identifier)
  ("<bbb-H>" mupad-send-identifier)
  ("<bbb-I>" mupad-send-identifier)
  ("<bbb-J>" mupad-send-identifier)
  ("<bbb-K>" mupad-send-identifier)
  ("<bbb-L>" mupad-send-identifier)
  ("<bbb-M>" mupad-send-identifier)
  ("<bbb-N>" mupad-send-identifier)
  ("<bbb-O>" mupad-send-identifier)
  ("<bbb-P>" mupad-send-identifier)
  ("<bbb-Q>" "Q_")
  ("<bbb-R>" "R_")
  ("<bbb-S>" mupad-send-identifier)
  ("<bbb-T>" mupad-send-identifier)
  ("<bbb-U>" mupad-send-identifier)
  ("<bbb-V>" mupad-send-identifier)
  ("<bbb-W>" mupad-send-identifier)
  ("<bbb-X>" mupad-send-identifier)
  ("<bbb-Y>" mupad-send-identifier)
  ("<bbb-Z>" "Z_")

  ("<lfloor>"  "floor(")
  ("<rfloor>"  ")")
  ("<lceil>"   "ceil(")
  ("<rceil>"   ")")

  ("<hbar>"    "`&hbar;`")
)
